
#include <FFGL.h>
#include <FFGLLib.h>
#include <FFGLFBO.h>

#include "FFGLPlugin.h"

#include <math.h>
#include <iostream>
#include <algorithm>

// FFGL Plugin Info Structure
// Insert the info that will appear on the host here 
/*!
	!!! Do not change fields marked with *
	*/

static CFFGLPluginInfo PluginInfo(
	Plugin::CreateInstance,					// * Create method
	"HV10",									// Plugin unique ID
	"Bumper",								// Plugin name
	1,										// * API major version number
	500,									// * API minor version number
	1,										// Plugin major version number
	000,									// Plugin minor version number
	FF_EFFECT,								// Plugin type FF_SOURCE or FF_EFFECT ( filters ) 
	"Bump harder and harder",		       	// Plugin description
	"Joris de Jong // Hybrid Visuals"		// About
	, 0										// *
	, NULL									// *
	);

Plugin::Plugin()
{
	// Input properties
	SetMinInputs( 1 );
	SetMaxInputs( 1 );


	scaling = new FFGLFloatParam();
	addFloatParameter( "Scaling", 0.5f, scaling );
	sequence = new FFGLFloatParam();
	addFloatParameter( "Sequence", 0.0f, sequence );
	speed = new FFGLFloatParam();
	addFloatParameter( "Speed", 0.25f, speed );
	previous = new FFGLEventParam();
	addEventParameter( "Previous", previous );
	next = new FFGLEventParam();
	addEventParameter( "Next", next );
	sustain = new FFGLFloatParam();
	addFloatParameter( "Sustain", 0.2f, sustain );
	release = new FFGLFloatParam();
	addFloatParameter( "Release", 0.8f, release );

	activeSequence = 0;
	activeStep = -1;

	nextDown = false;
	prevDown = false;

	chaser = ChaserManager();
}

Plugin::~Plugin()
{

}


FFResult Plugin::InitGL( const FFGLViewportStruct *vp )
{
	glGenTextures( 1, &image );
	glBindTexture( GL_TEXTURE_2D, image );
	glTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA, aliasing.width, aliasing.height, 0, GL_RGBA, GL_UNSIGNED_BYTE, aliasing.pixel_data );
	glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );
	glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_BORDER );
	glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_BORDER );
	glBindTexture( GL_TEXTURE_2D, 0 );

	return FF_SUCCESS;
}

FFResult Plugin::DeInitGL()
{
	glDeleteTextures( 1, &image );
	return FF_SUCCESS;
}

FFResult Plugin::ProcessOpenGL( ProcessOpenGLStruct *pGL )
{
	if ( pGL->numInputTextures < 1 )
		return FF_FAIL;

	if ( pGL->inputTextures[ 0 ] == NULL )
		return FF_FAIL;

	FFGLTextureStruct &texture = *(pGL->inputTextures[ 0 ]);

	chaser.update( slices, sequences, texture, scaling->value );

	//this will calculate the next step based on the plugin logic
	update();

	glEnable( GL_BLEND );
	glBlendFunc( GL_ONE, GL_ONE_MINUS_SRC_ALPHA );
	Renderer::bindTexture( texture.Handle );

	for ( auto step : playingSteps )
	{
		//apply attack and decay values
		float value = 1.0f / (1.0f - sustain->value - (1.0f - release->value)) * (step.second - sustain->value);
		value = fmax( 0.0f, value );
		value = fmin( 1.0f, value );

		glColor4f( value, value, value, value );

		for ( auto uid : step.first )
		{
			RenderCoords coords = chaser.getCoordsForSlice( uid );
			Renderer::drawTexturedShape( coords.tex, coords.verts );
		}
	}

	if ( aliased )
	{
		glColor4f( 0.2f, 0.2f, 0.2f, 1.0f );
        FFGLTextureStruct texImage = { aliasing.width, aliasing.height, aliasing.width, aliasing.height, image };
		Renderer::bindTexture( texImage.Handle );
		for ( int i = 0; i < (int)slices.size(); i++ )
		{
			RenderCoords aliasedCoords = chaser.getAntiAliasedCoordsForSlice( slices[ i ].uniqueId );
			Renderer::drawTexturedShape( aliasedCoords.tex, aliasedCoords.verts );
		}
	}

	Renderer::unbindTexture();
	glDisable( GL_BLEND );

	return FF_SUCCESS;
}

void Plugin::addActiveStep()
{
	//sanity check
	if ( activeStep > (int)sequences[ activeSequence ].steps.size() - 1 || activeStep < 0 )
		return;

	playingSteps[ sequences[ activeSequence ].steps[ activeStep ] ] = 1.0f;
}

void Plugin::update()
{
	activeSequence = (int)floor( sequence->value * (sequences.size() - 1) );
	if ( activeSequence < 0 )
		activeSequence = 0;

	//sanity check 
	if ( activeSequence >= (int)sequences.size() )
		return;

	if ( next->enable )
	{
		if ( !nextDown )
		{
			nextDown = true;
			activeStep += 1;
			if ( activeStep > (int)sequences[ activeSequence ].steps.size() - 1 )
				activeStep = 0;

			addActiveStep();
		}
	}
	else
		nextDown = false;

	if ( previous->enable )
	{
		if ( !prevDown )
		{
			prevDown = true;
			activeStep -= 1;
			if ( activeStep < 0 )
				activeStep = (int)sequences[ activeSequence ].steps.size() - 1;

			addActiveStep();
		}
	}
	else
		prevDown = false;

	

	//update the time in each step
	//and remove steps that have reached zero
	std::vector<Step> stepsToErase;
	for ( auto step : playingSteps )
	{
		playingSteps[step.first] = step.second - 0.1f * speed->value;
		if ( step.second <= 0.0f )
			stepsToErase.push_back( step.first );
	}

	for ( auto step : stepsToErase )
		playingSteps.erase( step );
}

char* Plugin::GetParameterDisplay( unsigned int index )
{
	unsigned int pType = m_pPlugin->GetParamType( index );

	if ( pType != FF_FAIL )
	{
		float fValue = m_pPlugin->GetFloatParameter( index );
		memset( s_Displayvalue, 0, 15 );

		if ( index == getParamIndex( sequence ) )
		{
			if ( activeSequence >= (int)sequences.size() )
				sprintf( s_Displayvalue, "Sequence %i", int( fminf( (float)activeSequence, 15.0f ) + 1.0f ) );
			else
			{
				std::string name = sequences[ activeSequence ].name;
				std::string nameShort = name.substr( 0, 15 );
				sprintf( s_Displayvalue, "%s", nameShort.c_str() );
			}
		}
		else if ( index == getParamIndex( scaling ) )
		{
			if ( scaling->value < 0.33 )
				sprintf( s_Displayvalue, "%s", "Mask" );
			else if ( scaling->value < 0.66 )
				sprintf( s_Displayvalue, "%s", "Fill" );
			else
				sprintf( s_Displayvalue, "%s", "Stretch" );
		}
		else
			sprintf( s_Displayvalue, "%f", fValue );

		return s_Displayvalue;
	}

	return NULL;
}







